//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// This file makes fixes to existing 3rd party libraries (jquery, jquery UI, MS Ajax)
// Review this file when upgrading versions.

// Helper method to check jquery [UI] version
// Set allowPreviousVersions to true if you wish to allow versions up to a particular milestone.
// If false or undefined, exact version is required.
(function ($) {
    var jqueryVersion = $.fn.jquery,
        jqueryUIVersion = $.ui.version;

    $.checkVersion = {
        _isValidVersion: function (expectedVersion, currentVersion, allowPreviousVersions) {
            var i, expectedVersionInfo, currentVersionInfo;

            if (allowPreviousVersions) {
                expectedVersionInfo = expectedVersion.split('.'),
                currentVersionInfo = currentVersion.split('.');
                for (i = 0; i < expectedVersionInfo.length; i++) {
                    if (expectedVersionInfo[i] > currentVersionInfo[i]) {
                        return true;
                    }
                    else if (expectedVersionInfo[i] < currentVersionInfo[i]) {
                        return false;
                    }
                }
                return true;
            }
            else {
                return expectedVersion === currentVersion;
            }
        },
        jquery: function (expectedVersion, allowPreviousVersions) {
            var isValidVersion = this._isValidVersion(expectedVersion, jqueryVersion, allowPreviousVersions);
            Sys.Debug.assert(isValidVersion, String.format('Invalid jquery version; expected: {0}, actual: {1}', expectedVersion, jqueryVersion));
            return isValidVersion;
        },
        jqueryUI: function (expectedVersion, allowPreviousVersions) {
            var isValidVersion = this._isValidVersion(expectedVersion, jqueryUIVersion, allowPreviousVersions);
            Sys.Debug.assert(isValidVersion, String.format('Invalid jquery UI version; expected: {0}, actual: {1}', expectedVersion, jqueryUIVersion));
            return isValidVersion;
        }
    };
})(jQuery);

// jQuery fix for camel-cased data === 0 (http://bugs.jquery.com/ticket/10016)
// Remove when upgrading to jQuery 1.6.3 or later
// Test case: open browse dialog and ensure nodes are indented correctly (see bug 781797)
(function ($) {
    if ($.checkVersion.jquery('1.6.2')) {

        $.data = function( elem, name, data, pvt /* Internal Use Only */ ) {
	        if ( !jQuery.acceptData( elem ) ) {
		        return;
	        }

	        var internalKey = jQuery.expando, getByName = typeof name === "string", thisCache,

		        // We have to handle DOM nodes and JS objects differently because IE6-7
		        // can't GC object references properly across the DOM-JS boundary
		        isNode = elem.nodeType,

		        // Only DOM nodes need the global jQuery cache; JS object data is
		        // attached directly to the object so GC can occur automatically
		        cache = isNode ? jQuery.cache : elem,

		        // Only defining an ID for JS objects if its cache already exists allows
		        // the code to shortcut on the same path as a DOM node with no cache
		        id = isNode ? elem[ jQuery.expando ] : elem[ jQuery.expando ] && jQuery.expando;

	        // Avoid doing any more work than we need to when trying to get data on an
	        // object that has no data at all
	        if ( (!id || (pvt && id && !cache[ id ][ internalKey ])) && getByName && data === undefined ) {
		        return;
	        }

	        if ( !id ) {
		        // Only DOM nodes need a new unique ID for each element since their data
		        // ends up in the global cache
		        if ( isNode ) {
			        elem[ jQuery.expando ] = id = ++jQuery.uuid;
		        } else {
			        id = jQuery.expando;
		        }
	        }

	        if ( !cache[ id ] ) {
		        cache[ id ] = {};

		        // TODO: This is a hack for 1.5 ONLY. Avoids exposing jQuery
		        // metadata on plain JS objects when the object is serialized using
		        // JSON.stringify
		        if ( !isNode ) {
			        cache[ id ].toJSON = jQuery.noop;
		        }
	        }

	        // An object can be passed to jQuery.data instead of a key/value pair; this gets
	        // shallow copied over onto the existing cache
	        if ( typeof name === "object" || typeof name === "function" ) {
		        if ( pvt ) {
			        cache[ id ][ internalKey ] = jQuery.extend(cache[ id ][ internalKey ], name);
		        } else {
			        cache[ id ] = jQuery.extend(cache[ id ], name);
		        }
	        }

	        thisCache = cache[ id ];

	        // Internal jQuery data is stored in a separate object inside the object's data
	        // cache in order to avoid key collisions between internal data and user-defined
	        // data
	        if ( pvt ) {
		        if ( !thisCache[ internalKey ] ) {
			        thisCache[ internalKey ] = {};
		        }

		        thisCache = thisCache[ internalKey ];
	        }

	        if ( data !== undefined ) {
		        thisCache[ jQuery.camelCase( name ) ] = data;
	        }

	        // TODO: This is a hack for 1.5 ONLY. It will be removed in 1.6. Users should
	        // not attempt to inspect the internal events object using jQuery.data, as this
	        // internal data object is undocumented and subject to change.
	        if ( name === "events" && !thisCache[name] ) {
		        return thisCache[ internalKey ] && thisCache[ internalKey ].events;
	        }

	        // This is ported from jQuery 1.6.3 (http://bugs.jquery.com/ticket/10016)
	        // Check for both converted-to-camel and non-converted data property names
	        // If a data property was specified
	        if ( getByName ) {

		        // First Try to find as-is property data
		        ret = thisCache[ name ];

		        // Test for null|undefined property data
		        if ( ret == null ) {

			        // Try to find the camelCased property
			        ret = thisCache[ jQuery.camelCase( name ) ];
		        }
	        } else {
		        ret = thisCache;
	        }

	        return ret;
        };
    }
})(jQuery);

// jQuery fix to close topmost modal dialog instead of all dialogs
// Remove when upgrading to jQuery UI 1.8.17 or later
// Test case: open stacked modal dialogs and hit esc key to close topmost
(function ($) {
    if ($.checkVersion.jqueryUI('1.8.17', true)) {

        var oldOverlayCreate = $.ui.dialog.overlay.create;
        $.ui.dialog.overlay.create = function(dialog) {
            var overlay = oldOverlayCreate.call(this, dialog);
            overlay.data('current-dialog', dialog);

            if (this.instances.length === 1) {

                // allow closing by pressing the escape key
                $(document).unbind('keydown.dialog-overlay');
                $(document).bind('keydown.dialog-overlay', function(event) {
                    var maxZ = 0, currentOverlay, dialog;
                            
                    if (!event.isDefaultPrevented() && event.keyCode && event.keyCode === $.ui.keyCode.ESCAPE) {
                        // When more than 1 dialogs stacked together we want correct dialog to close not the very first one.
                        $.each($.ui.dialog.overlay.instances, function () {
                            var zIndex = $(this).css('z-index');
                            if (zIndex > maxZ) {
                                currentOverlay = this;
                                maxZ = zIndex;
                            }
                        });

                        if (currentOverlay && (dialog = currentOverlay.data("current-dialog")) && dialog.options.closeOnEscape) {
                            dialog.close(event);
                            return false;
                        }
                    }
                });
            }
            return overlay;
        };
    }
})(jQuery);

// jQuery fix to keep tab focus inside dialog
// and to prevent shift-enter from activating background button
// Modify when upgrading to jQuery UI 1.8.17 or later
// Test case: open a dialog with a button, then hit Shift+Enter in IE to do nothing
(function ($) {
    if ($.checkVersion.jqueryUI('1.8.17', true)) {

        var dialogWidget = $.ui.dialog;
        var oldDialogOpen = $.ui.dialog.prototype.open;
        var oldDialogClose = $.ui.dialog.prototype.close;
        $.widget("ui.dialog", $.extend({}, $.ui.dialog.prototype, {
            open: function () {
                var self = this, dialog = oldDialogOpen.apply(this, arguments);
                if (this.options.modal) {
                    $.each(this.uiDialog.data('events').keypress, function (i, elem) {
                        if (elem.namespace === 'ui-dialog') {
                            self.uiDialog.bind('keydown.ui-dialog', function (event) {
                                if (event.keyCode === $.ui.keyCode.ENTER && event.shiftKey) {
		                            return false; // IE bug: without this, Shift+Enter triggers click on button (if applicable) used to open dialog
		                        }

                                return elem.handler.call(this, event);
                            });

                            self.uiDialog.unbind('keypress.ui-dialog');
                            return false;
                        }
                    });
                }
                return dialog;
            },
            close: function () {
                this.uiDialog.unbind('keydown.ui-dialog');
                return oldDialogClose.apply(this, arguments);
            }
        }));
        $.extend($.ui.dialog, dialogWidget);
    }
})(jQuery);

// jQuery fix for tab accessibility, based on 1.8.14.  Keep in sync with _tabify function.
// Remove when http://bugs.jqueryui.com/ticket/3079 gets resolved
// Test case 1: in IE, select a tab with keyboard and hit enter to maintain focus
(function ($) {
    if ($.checkVersion.jqueryUI('1.8.14')) {

        var tabsWidget = $.ui.tabs;
        var oldTabify = $.ui.tabs.prototype._tabify;
        $.widget("ui.tabs", $.extend({}, $.ui.tabs.prototype, {
            _tabify: function () {
                oldTabify.apply(this, arguments);

                var self = this, o = this.options;

		        // set up animations
		        var hideFx, showFx;
		        if ( o.fx ) {
			        if ( $.isArray( o.fx ) ) {
				        hideFx = o.fx[ 0 ];
				        showFx = o.fx[ 1 ];
			        } else {
				        hideFx = showFx = o.fx;
			        }
		        }

		        // Reset certain styles left over from animation
		        // and prevent IE's ClearType bug...
		        function resetStyle( $el, fx ) {
			        $el.css( "display", "" );
			        if ( !$.support.opacity && fx.opacity ) {
				        $el[ 0 ].style.removeAttribute( "filter" );
			        }
		        }

		        // Show a tab...
		        var showTab = showFx
			        ? function( clicked, $show ) {
				        $( clicked ).closest( "li" ).addClass( "ui-tabs-selected ui-state-active" );
				        $show.hide().removeClass( "ui-tabs-hide" ) // avoid flicker that way
					        .animate( showFx, showFx.duration || "normal", function() {
						        resetStyle( $show, showFx );
						        self._trigger( "show", null, self._ui( clicked, $show[ 0 ] ) );
					        });
			        }
			        : function( clicked, $show ) {
				        $( clicked ).closest( "li" ).addClass( "ui-tabs-selected ui-state-active" );
				        $show.removeClass( "ui-tabs-hide" );
				        self._trigger( "show", null, self._ui( clicked, $show[ 0 ] ) );
			        };

		        // Hide a tab, $show is optional...
		        var hideTab = hideFx
			        ? function( clicked, $hide ) {
				        $hide.animate( hideFx, hideFx.duration || "normal", function() {
					        self.lis.removeClass( "ui-tabs-selected ui-state-active" );
					        $hide.addClass( "ui-tabs-hide" );
					        resetStyle( $hide, hideFx );
					        self.element.dequeue( "tabs" );
				        });
			        }
			        : function( clicked, $hide, $show ) {
				        self.lis.removeClass( "ui-tabs-selected ui-state-active" );
				        $hide.addClass( "ui-tabs-hide" );
				        self.element.dequeue( "tabs" );
			        };

                this.anchors.unbind(o.event + '.tabs');

                // attach tab event handler, unbind to avoid duplicates from former tabifying...
                this.anchors.bind( o.event + ".tabs", function() {
                    var el = this,
                        $li = $(el).closest( "li" ),
                        $hide = self.panels.filter( ":not(.ui-tabs-hide)" ),
                        $show = self.element.find( self._sanitizeSelector( el.hash ) );

                    // If tab is already selected and not collapsible or tab disabled or
                    // or is already loading or click callback returns false stop here.
                    // Check if click handler returns false last so that it is not executed
                    // for a disabled or loading tab!
                    if ( ( $li.hasClass( "ui-tabs-selected" ) && !o.collapsible) ||
                        $li.hasClass( "ui-state-disabled" ) ||
                        $li.hasClass( "ui-state-processing" ) ||
                        self.panels.filter( ":animated" ).length ||
                        self._trigger( "select", null, self._ui( this, $show[ 0 ] ) ) === false ) {
                        // There is an open bug on tabs keyboard accessibility
                        // http://bugs.jqueryui.com/ticket/3079
                        // this.blur();
                        return false;
                    }

                    o.selected = self.anchors.index( this );

                    self.abort();

                    // if tab may be closed
                    if ( o.collapsible ) {
                        if ( $li.hasClass( "ui-tabs-selected" ) ) {
                            o.selected = -1;

                            if ( o.cookie ) {
            	                self._cookie( o.selected, o.cookie );
                            }

                            self.element.queue( "tabs", function() {
            	                hideTab( el, $hide );
                            }).dequeue( "tabs" );

                            this.blur();
                            return false;
                        } else if ( !$hide.length ) {
                            if ( o.cookie ) {
            	                self._cookie( o.selected, o.cookie );
                            }

                            self.element.queue( "tabs", function() {
            	                showTab( el, $show );
                            });

                            // TODO make passing in node possible, see also http://dev.jqueryui.com/ticket/3171
                            self.load( self.anchors.index( this ) );

                            this.blur();
                            return false;
                        }
                    }

                    if ( o.cookie ) {
                        self._cookie( o.selected, o.cookie );
                    }

                    // show new tab
                    if ( $show.length ) {
                        if ( $hide.length ) {
                            self.element.queue( "tabs", function() {
            	                hideTab( el, $hide );
                            });
                        }
                        self.element.queue( "tabs", function() {
                            showTab( el, $show );
                        });

                        self.load( self.anchors.index( this ) );
                    } else {
                        throw "jQuery UI Tabs: Mismatching fragment identifier.";
                    }

                    // Prevent IE from keeping other link focussed when using the back button
                    // and remove dotted border from clicked link. This is controlled via CSS
                    // in modern browsers; blur() removes focus from address bar in Firefox
                    // which can become a usability and annoying problem with tabs('rotate').
                    //if ($.browser.msie) {
                        // There is an open bug on tabs keyboard accessibility
                        // http://bugs.jqueryui.com/ticket/3079
                        //this.blur();
                    //}
                });

		        // disable click in any case
		        this.anchors.bind( "click.tabs", function(){
			        return false;
		        });
            }
        }));
        $.extend($.ui.tabs, tabsWidget);
    }
})(jQuery);

if (Sys && Sys._Application) {
    if ((Sys.Browser.agent === Sys.Browser.InternetExplorer) && (Sys.Browser.documentMode < 8)) {

        // We do not support IE7 and below. IE8 will appear to be IE7 when run in compatibility mode (the 
        // default for sites in the Intranet zone), but IE8 natively supports history state, so prevent the 
        // legacy/IFrame trick from being used at all by removing that block of code from the
        // original MS Ajax implementation

        Sys._Application.prototype._ensureHistory = function Sys$_Application$_ensureHistory() {
            if (!this._historyInitialized && this._enableHistory) {

                this._timerHandler = Function.createDelegate(this, this._onIdle);
                this._timerCookie = window.setTimeout(this._timerHandler, 100);

                try {
                    this._initialState = this._deserializeState(this.get_stateString());
                } catch (e) { }

                this._historyInitialized = true;
            }
        };
    }
}
// SIG // Begin signature block
// SIG // MIIaswYJKoZIhvcNAQcCoIIapDCCGqACAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFE7xTM64rOM1
// SIG // ylhf2FH11u/hftanoIIVgjCCBMMwggOroAMCAQICEzMA
// SIG // AAArOTJIwbLJSPMAAAAAACswDQYJKoZIhvcNAQEFBQAw
// SIG // dzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgUENBMB4XDTEyMDkwNDIx
// SIG // MTIzNFoXDTEzMTIwNDIxMTIzNFowgbMxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsT
// SIG // Hm5DaXBoZXIgRFNFIEVTTjpDMEY0LTMwODYtREVGODEl
// SIG // MCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
// SIG // dmljZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBAKa2MA4DZa5QWoZrhZ9IoR7JwO5eSQeF4HCWfL65
// SIG // X2JfBibTizm7GCKlLpKt2EuIOhqvm4OuyF45jMIyexZ4
// SIG // 7Tc4OvFi+2iCAmjs67tAirH+oSw2YmBwOWBiDvvGGDhv
// SIG // sJLWQA2Apg14izZrhoomFxj/sOtNurspE+ZcSI5wRjYm
// SIG // /jQ1qzTh99rYXOqZfTG3TR9X63zWlQ1mDB4OMhc+LNWA
// SIG // oc7r95iRAtzBX/04gPg5f11kyjdcO1FbXYVfzh4c+zS+
// SIG // X+UoVXBUnLjsfABVRlsomChWTOHxugkZloFIKjDI9zMg
// SIG // bOdpw7PUw07PMB431JhS1KkjRbKuXEFJT7RiaJMCAwEA
// SIG // AaOCAQkwggEFMB0GA1UdDgQWBBSlGDNTP5VgoUMW747G
// SIG // r9Irup5Y0DAfBgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7
// SIG // syuwwzWzDzBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9NaWNyb3NvZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsG
// SIG // AQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMIMA0GCSqGSIb3DQEBBQUAA4IBAQB+zLB75S++
// SIG // 51a1z3PbqlLRFjnGtM361/4eZbXnSPObRogFZmomhl7+
// SIG // h1jcxmOOOID0CEZ8K3OxDr9BqsvHqpSkN/BkOeHF1fnO
// SIG // B86r5CXwaa7URuL+ZjI815fFMiH67holoF4MQiwRMzqC
// SIG // g/3tHbO+zpGkkSVxuatysJ6v5M8AYolwqbhKUIzuLyJk
// SIG // pajmTWuVLBx57KejMdqQYJCkbv6TAg0/LCQNxmomgVGD
// SIG // ShC7dWNEqmkIxgPr4s8L7VY67O9ypwoM9ADTIrivInKz
// SIG // 58ScCyiggMrj4dc5ZjDnRhcY5/qC+lkLeryoDf4c/wOL
// SIG // Y7JNEgIjTy2zhYQ74qFH6M8VMIIE7DCCA9SgAwIBAgIT
// SIG // MwAAALARrwqL0Duf3QABAAAAsDANBgkqhkiG9w0BAQUF
// SIG // ADB5MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSMwIQYDVQQDExpN
// SIG // aWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQTAeFw0xMzAx
// SIG // MjQyMjMzMzlaFw0xNDA0MjQyMjMzMzlaMIGDMQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYD
// SIG // VQQDExVNaWNyb3NvZnQgQ29ycG9yYXRpb24wggEiMA0G
// SIG // CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDor1yiIA34
// SIG // KHy8BXt/re7rdqwoUz8620B9s44z5lc/pVEVNFSlz7SL
// SIG // qT+oN+EtUO01Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Un
// SIG // m/P4LZ5BNisLQPu+O7q5XHWTFlJLyjPFN7Dz636o9UEV
// SIG // XAhlHSE38Cy6IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpO
// SIG // oPXJCiHiquMBNkf9L4JqgQP1qTXclFed+0vUDoLbOI8S
// SIG // /uPWenSIZOFixCUuKq6dGB8OHrbCryS0DlC83hyTXEmm
// SIG // ebW22875cHsoAYS4KinPv6kFBeHgD3FN/a1cI4Mp68fF
// SIG // SsjoJ4TTfsZDC5UABbFPZXHFAgMBAAGjggFgMIIBXDAT
// SIG // BgNVHSUEDDAKBggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGm
// SIG // WjNN2pgHgP+EHr6H+XIyQfIwUQYDVR0RBEowSKRGMEQx
// SIG // DTALBgNVBAsTBE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRm
// SIG // YWYwYjcxLWFkMzctNGFhMy1hNjcxLTc2YmMwNTIzNDRh
// SIG // ZDAfBgNVHSMEGDAWgBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWND
// SIG // b2RTaWdQQ0FfMDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUH
// SIG // AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNpZ1BD
// SIG // QV8wOC0zMS0yMDEwLmNydDANBgkqhkiG9w0BAQUFAAOC
// SIG // AQEAMdduKhJXM4HVncbr+TrURE0Inu5e32pbt3nPApy8
// SIG // dmiekKGcC8N/oozxTbqVOfsN4OGb9F0kDxuNiBU6fNut
// SIG // zrPJbLo5LEV9JBFUJjANDf9H6gMH5eRmXSx7nR2pEPoc
// SIG // sHTyT2lrnqkkhNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2y
// SIG // RPnwPJNtQtjofOYXoJtoaAko+QKX7xEDumdSrcHps3Om
// SIG // 0mPNSuI+5PNO/f+h4LsCEztdIN5VP6OukEAxOHUoXgSp
// SIG // Rm3m9Xp5QL0fzehF1a7iXT71dcfmZmNgzNWahIeNJDD3
// SIG // 7zTQYx2xQmdKDku/Og7vtpU6pzjkJZIIpohmgjCCBbww
// SIG // ggOkoAMCAQICCmEzJhoAAAAAADEwDQYJKoZIhvcNAQEF
// SIG // BQAwXzETMBEGCgmSJomT8ixkARkWA2NvbTEZMBcGCgmS
// SIG // JomT8ixkARkWCW1pY3Jvc29mdDEtMCsGA1UEAxMkTWlj
// SIG // cm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
// SIG // MB4XDTEwMDgzMTIyMTkzMloXDTIwMDgzMTIyMjkzMlow
// SIG // eTELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWlj
// SIG // cm9zb2Z0IENvZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqG
// SIG // SIb3DQEBAQUAA4IBDwAwggEKAoIBAQCycllcGTBkvx2a
// SIG // YCAgQpl2U2w+G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/
// SIG // 3sJCTiPVcgDbNVcKicquIEn08GisTUuNpb15S3GbRwfa
// SIG // /SXfnXWIz6pzRH/XgdvzvfI2pMlcRdyvrT3gKGiXGqel
// SIG // cnNW8ReU5P01lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJp
// SIG // L9oZC/6SdCnidi9U3RQwWfjSjWL9y8lfRjFQuScT5EAw
// SIG // z3IpECgixzdOPaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn
// SIG // 0i1i8UU956wIAPZGoZ7RW4wmU+h6qkryRs83PDietHdc
// SIG // pReejcsRj1Y8wawJXwPTAgMBAAGjggFeMIIBWjAPBgNV
// SIG // HRMBAf8EBTADAQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJ
// SIG // Ny4zFha5TJoKHzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGC
// SIG // NxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQU/dExTtMm
// SIG // ipXhmGA7qDFvpjy82C0wGQYJKwYBBAGCNxQCBAweCgBT
// SIG // AHUAYgBDAEEwHwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8
// SIG // KuEKU5VZ5KQwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDov
// SIG // L2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVj
// SIG // dHMvbWljcm9zb2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUF
// SIG // BwEBBEgwRjBEBggrBgEFBQcwAoY4aHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRS
// SIG // b290Q2VydC5jcnQwDQYJKoZIhvcNAQEFBQADggIBAFk5
// SIG // Pn8mRq/rb0CxMrVq6w4vbqhJ9+tfde1MOy3XQ60L/svp
// SIG // LTGjI8x8UJiAIV2sPS9MuqKoVpzjcLu4tPh5tUly9z7q
// SIG // QX/K4QwXaculnCAt+gtQxFbNLeNK0rxw56gNogOlVuC4
// SIG // iktX8pVCnPHz7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y
// SIG // 4k74jKHK6BOlkU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wd
// SIG // zaKMvSeyeWNWRKJRzfnpo1hW3ZsCRUQvX/TartSCMm78
// SIG // pJUT5Otp56miLL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q7
// SIG // 0eFW6NB4lhhcyTUWX92THUmOLb6tNEQc7hAVGgBd3TVb
// SIG // Ic6YxwnuhQ6MT20OE049fClInHLR82zKwexwo1eSV32U
// SIG // jaAbSANa98+jZwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKN
// SIG // MxZlHg6K3RDeZPRvzkbU0xfpecQEtNP7LN8fip6sCvsT
// SIG // J0Ct5PnhqX9GuwdgR2VgQE6wQuxO7bN2edgKNAltHIAx
// SIG // H+IOVN3lofvlRxCtZJj/UBYufL8FIXrilUEnacOTj5XJ
// SIG // jdibIa4NXJzwoq6GaIMMai27dmsAHZat8hZ79haDJLmI
// SIG // z2qoRzEvmtzjcT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq
// SIG // /2mbluIQqBC0N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZo
// SIG // NAAAAAAAHDANBgkqhkiG9w0BAQUFADBfMRMwEQYKCZIm
// SIG // iZPyLGQBGRYDY29tMRkwFwYKCZImiZPyLGQBGRYJbWlj
// SIG // cm9zb2Z0MS0wKwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBD
// SIG // ZXJ0aWZpY2F0ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1
// SIG // MzA5WhcNMjEwNDAzMTMwMzA5WjB3MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBQQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCfoWyx39tIkip8ay4Z4b3i48WZUSNQrc7d
// SIG // GE4kD+7Rp9FMrXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr
// SIG // 6Hu97IkHD/cOBJjwicwfyzMkh53y9GccLPx754gd6udO
// SIG // o6HBI1PKjfpFzwnQXq/QsEIEovmmbJNn1yjcRlOwhtDl
// SIG // KEYuJ6yGT1VSDOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd+
// SIG // +NIT8wi3U21StEWQn0gASkdmEScpZqiX5NMGgUqi+YSn
// SIG // EUcUCYKfhO1VeP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68e
// SIG // eEExd8yb3zuDk6FhArUdDbH895uyAc4iS1T/+QXDwiAL
// SIG // AgMBAAGjggGrMIIBpzAPBgNVHRMBAf8EBTADAQH/MB0G
// SIG // A1UdDgQWBBQjNPjZUkZwCu1A+3b7syuwwzWzDzALBgNV
// SIG // HQ8EBAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwgZgGA1Ud
// SIG // IwSBkDCBjYAUDqyCYEBWJ5flJRP8KuEKU5VZ5KShY6Rh
// SIG // MF8xEzARBgoJkiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJ
// SIG // k/IsZAEZFgltaWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jv
// SIG // c29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eYIQ
// SIG // ea0WoUqgpa1Mc1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BB
// SIG // hj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
// SIG // bC9wcm9kdWN0cy9taWNyb3NvZnRyb290Y2VydC5jcmww
// SIG // VAYIKwYBBQUHAQEESDBGMEQGCCsGAQUFBzAChjhodHRw
// SIG // Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01p
// SIG // Y3Jvc29mdFJvb3RDZXJ0LmNydDATBgNVHSUEDDAKBggr
// SIG // BgEFBQcDCDANBgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wD
// SIG // RDbd6bStd9vOeVFNAbEudHFbbQwTq86+e4+4LtQSooxt
// SIG // YrhXAstOIBNQmd16QOJXu69YmhzhHQGGrLt48ovQ7DsB
// SIG // 7uK+jwoFyI1I4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mR
// SIG // KiQicPv2/OR4mS4N9wficLwYTp2OawpylbihOZxnLcVR
// SIG // DupiXD8WmIsgP+IHGjL5zDFKdjE9K3ILyOpwPf+FChPf
// SIG // wgphjvDXuBfrTot/xTUrXqO/67x9C0J71FNyIe4wyrt4
// SIG // ZVxbARcKFA7S2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGys
// SIG // OUzU9nm/qhh6YinvopspNAZ3GmLJPR5tH4LwC8csu89D
// SIG // s+X57H2146SodDW4TsVxIxImdgs8UoxxWkZDFLyzs7BN
// SIG // Z8ifQv+AeSGAnhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU
// SIG // 2DKATCYqSCRfWupW76bemZ3KOm+9gSd0BhHudiG/m4LB
// SIG // J1S2sWo9iaF2YbRuoROmv6pH8BJv/YoybLL+31HIjCPJ
// SIG // Zr2dHYcSZAI9La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCl
// SIG // eKuzoJZ1GtmShxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/J
// SIG // mu5J4PcBZW+JC33Iacjmbuqnl84xKf8OxVtc2E0bodj6
// SIG // L54/LlUWa8kTo/0xggSdMIIEmQIBATCBkDB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQQITMwAAALARrwqL0Duf3QAB
// SIG // AAAAsDAJBgUrDgMCGgUAoIG2MBkGCSqGSIb3DQEJAzEM
// SIG // BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgor
// SIG // BgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBR1h7Vh++JS
// SIG // 0xD0bqgV7C/AOHWwXjBWBgorBgEEAYI3AgEMMUgwRqAs
// SIG // gCoAagBxAHUAZQByAHkALQBmAGkAeABlAHMALgBkAGUA
// SIG // YgB1AGcALgBqAHOhFoAUaHR0cDovL21pY3Jvc29mdC5j
// SIG // b20wDQYJKoZIhvcNAQEBBQAEggEAmBBbGuX3tiYfBX0k
// SIG // B2gfSwXQ3RFeIRigNOsLsedmKXq2dHJ9QSsmioW94moZ
// SIG // KBf65NXQlLXGdK9IQ7yptegCNBQyqKIOtXJaQ2LcZlx6
// SIG // lNdnKH0Xr5WdXIbrqaTra7SvlxHm9249mVJ0QjyCNe6L
// SIG // AUFmH9aOxJT2BohoJrpkM5pHj8II+B0Udo7DJIoXncN6
// SIG // 9bBrDgI7AaQiI+MRC6IsL/Y24vJ3bzgQjf2/gpWvkt0c
// SIG // W/BMoXeMPGGfIfNx3oi+G9x0a6r/SX3rD6/yz5mI8oHl
// SIG // Q6pgPYD6H0uy9Se9RQ6EAJr8Yfb1j62XCDfpVYwsuFw2
// SIG // L3a4vWJW4XIBImUYGaGCAigwggIkBgkqhkiG9w0BCQYx
// SIG // ggIVMIICEQIBATCBjjB3MQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
// SIG // ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
// SIG // MSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQ
// SIG // Q0ECEzMAAAArOTJIwbLJSPMAAAAAACswCQYFKw4DAhoF
// SIG // AKBdMBgGCSqGSIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJ
// SIG // KoZIhvcNAQkFMQ8XDTEzMDMxNTA2MzM1NVowIwYJKoZI
// SIG // hvcNAQkEMRYEFCFcsOoO0Bp94bVCGrnBqhHaD/cpMA0G
// SIG // CSqGSIb3DQEBBQUABIIBACoL2f/zMm0cZxxSlUHyeyGH
// SIG // 2c/IW4SGc4K7pX86kg+CFBh+93REQfsd23/GH83sqt9Z
// SIG // PQGdYIoAm0nuwHrVdVy88oSzExwiULAuqlzYnMzIbgDR
// SIG // ctwDdwXAqTVdncndxGKZ79E4IFnxpTVXH2+MiVnuPKJo
// SIG // xRBQ6nhM46MTUSGjFVeyX0myBNQcNy3v/6OOyvbNszwf
// SIG // +CT3JNJrtXrmaRUdM2JtlZpZ0fsKcbXwNquB3uX0+tKL
// SIG // sIvmGaLTOKrkKozGRjHoNqP/0kUIu2part+na2qDeZD1
// SIG // 7f/WDnnsCzoVTz7S/GWPDcFmrtNJcgoS5JF/kZV3rORo
// SIG // ldA3wTl7Y8s=
// SIG // End signature block
